#!/bin/bash

# Script to delete all hidden files in a directory
# Hidden files are files that start with a dot (.)

# Function to display usage
usage() {
    echo "Usage: $0 [OPTIONS] [DIRECTORY]"
    echo ""
    echo "Delete all hidden files in the specified directory (or current directory if not specified)."
    echo ""
    echo "OPTIONS:"
    echo "  -d, --dry-run    Show what would be deleted without actually deleting"
    echo "  -f, --force      Delete without confirmation prompt"
    echo "  -r, --recursive  Process all subdirectories recursively"
    echo "  -h, --help       Show this help message"
    echo ""
    echo "Examples:"
    echo "  $0                    # Delete hidden files in current directory"
    echo "  $0 /path/to/dir       # Delete hidden files in specified directory"
    echo "  $0 -r /path/to/dir    # Delete hidden files in directory and all subdirectories"
    echo "  $0 -d /path/to/dir    # Dry run: show what would be deleted"
    echo "  $0 -f /path/to/dir    # Force delete without confirmation"
    exit 1
}

# Default values
DRY_RUN=false
FORCE=false
RECURSIVE=false
TARGET_DIR=""

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        -d|--dry-run)
            DRY_RUN=true
            shift
            ;;
        -f|--force)
            FORCE=true
            shift
            ;;
        -r|--recursive)
            RECURSIVE=true
            shift
            ;;
        -h|--help)
            usage
            ;;
        -*)
            echo "Unknown option: $1"
            usage
            ;;
        *)
            if [ -z "$TARGET_DIR" ]; then
                TARGET_DIR="$1"
            else
                echo "Error: Multiple directories specified"
                usage
            fi
            shift
            ;;
    esac
done

# Use current directory if no directory specified
if [ -z "$TARGET_DIR" ]; then
    TARGET_DIR="."
fi

# Check if directory exists
if [ ! -d "$TARGET_DIR" ]; then
    echo "Error: Directory '$TARGET_DIR' does not exist."
    exit 1
fi

# Convert to absolute path
TARGET_DIR=$(cd "$TARGET_DIR" && pwd)

# Function to find hidden files in a directory
find_hidden_files() {
    local dir="$1"
    find "$dir" -maxdepth 1 -name ".*" -type f 2>/dev/null | grep -v "^$"
}

# Function to count hidden files in a directory
count_hidden_files() {
    local dir="$1"
    find_hidden_files "$dir" | grep -v "^$" | wc -l
}

# Function to display hidden files (for preview)
display_hidden_files() {
    local dir="$1"
    local hidden_files=$(find_hidden_files "$dir")
    local file_count=$(count_hidden_files "$dir")
    
    if [ "$file_count" -gt 0 ]; then
        if [ "$RECURSIVE" = true ] && [ "$dir" != "$TARGET_DIR" ]; then
            echo ""
            echo "=== $dir ==="
        fi
        echo "Found $file_count hidden file(s) in '$dir':"
        echo "$hidden_files" | grep -v "^$" | while read -r file; do
            echo "  - $file"
        done
    fi
}

# Function to delete hidden files in a directory
delete_hidden_files() {
    local dir="$1"
    local hidden_files=$(find_hidden_files "$dir")
    local deleted_count=0
    local failed_count=0
    
    if [ -z "$hidden_files" ] || [ -z "$(echo "$hidden_files" | grep -v "^$")" ]; then
        return 0
    fi
    
    if [ "$RECURSIVE" = true ] && [ "$dir" != "$TARGET_DIR" ]; then
        echo ""
        echo "=== Processing: $dir ==="
    fi
    
    while IFS= read -r file; do
        if [ -n "$file" ]; then
            if rm -f "$file" 2>/dev/null; then
                echo "Deleted: $file"
                ((deleted_count++))
            else
                echo "Failed to delete: $file"
                ((failed_count++))
            fi
        fi
    done <<< "$hidden_files"
    
    return $failed_count
}

# Collect all directories to process
if [ "$RECURSIVE" = true ]; then
    # Process target directory and all subdirectories
    DIRS_TO_PROCESS=("$TARGET_DIR")
    while IFS= read -r subdir; do
        DIRS_TO_PROCESS+=("$subdir")
    done < <(find "$TARGET_DIR" -mindepth 1 -type d 2>/dev/null)
else
    # Process only target directory
    DIRS_TO_PROCESS=("$TARGET_DIR")
fi

# First pass: display all files that would be deleted
TOTAL_FILES=0
for dir in "${DIRS_TO_PROCESS[@]}"; do
    display_hidden_files "$dir"
    file_count=$(count_hidden_files "$dir")
    TOTAL_FILES=$((TOTAL_FILES + file_count))
done

# Check if any files were found
if [ "$TOTAL_FILES" -eq 0 ]; then
    echo "No hidden files found."
    exit 0
fi

# Dry run mode
if [ "$DRY_RUN" = true ]; then
    echo ""
    echo "DRY RUN: No files were deleted."
    exit 0
fi

# Confirmation prompt (unless force is enabled)
if [ "$FORCE" = false ]; then
    echo ""
    read -p "Are you sure you want to delete these files? (y/N): " -n 1 -r
    echo ""
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "Operation cancelled."
        exit 0
    fi
fi

# Second pass: actually delete the files
TOTAL_DELETED=0
TOTAL_FAILED=0
for dir in "${DIRS_TO_PROCESS[@]}"; do
    delete_hidden_files "$dir"
    failed=$?
    TOTAL_FAILED=$((TOTAL_FAILED + failed))
done

# Calculate total deleted (approximate)
TOTAL_DELETED=$((TOTAL_FILES - TOTAL_FAILED))

echo ""
if [ $TOTAL_FAILED -eq 0 ]; then
    echo "Successfully deleted $TOTAL_DELETED file(s)."
else
    echo "Deleted $TOTAL_DELETED file(s), $TOTAL_FAILED failed."
fi

# Successfully deleted 952 file(s).