# Modelo de Base de Datos - Todoyeso

## Información General

- **Tipo de Base de Datos**: SQLite
- **Archivo**: `db.sqlite3`
- **ORM**: SQLAlchemy (Declarative Base)
- **Ubicación**: `/home/crepu/Documents/todoyeso/scripts/descriptions/db.sqlite3`

## Estructura de Tablas

### Tabla: `category`

Almacena las categorías de productos de yeso.

| Campo | Tipo | Restricciones | Descripción |
|-------|------|---------------|-------------|
| `id` | INTEGER | PRIMARY KEY, AUTO INCREMENT | Identificador único de la categoría |
| `name` | VARCHAR(255) | NOT NULL | Nombre legible de la categoría (ej: "Figuras Decorativas") |
| `dir_name` | VARCHAR(255) | NOT NULL | Nombre del directorio asociado (ej: "figuras_decorativas") |
| `hay_fotos` | BOOLEAN | DEFAULT TRUE | Indica si la categoría tiene fotos asociadas |
| `hay_tamanos` | BOOLEAN | DEFAULT TRUE | Indica si la categoría tiene tamaños disponibles |

**Relaciones**:
- One-to-Many con `image` (una categoría puede tener múltiples imágenes)
- Cascade: `all, delete-orphan` (al eliminar una categoría, se eliminan sus imágenes)

**Índices**:
- `id` (PRIMARY KEY)
- `dir_name` (usado para búsquedas)

### Tabla: `image`

Almacena las imágenes de productos con sus descripciones generadas por IA.

| Campo | Tipo | Restricciones | Descripción |
|-------|------|---------------|-------------|
| `id` | INTEGER | PRIMARY KEY, AUTO INCREMENT | Identificador único de la imagen |
| `ia_name` | VARCHAR(255) | NOT NULL | Nombre del producto generado por IA |
| `file_name` | VARCHAR(255) | NOT NULL | Ruta completa del archivo de imagen |
| `description` | VARCHAR(1000) | NOT NULL | Descripción detallada del producto generada por IA (400-800 caracteres) |
| `category_id` | INTEGER | FOREIGN KEY, NOT NULL | Referencia a `category.id` |

**Relaciones**:
- Many-to-One con `category` (cada imagen pertenece a una categoría)
- Foreign Key: `category_id` → `category.id`
- Cascade: `all, delete-orphan` (si se elimina la categoría, se eliminan las imágenes)
- `single_parent=True` (una imagen solo puede pertenecer a una categoría)

**Relaciones** (extendidas):
- One-to-One con `product_details` (cada imagen puede tener detalles de producto asociados)
- Nota: La relación bidireccional se define en `database_extended.py`

**Índices**:
- `id` (PRIMARY KEY)
- `category_id` (FOREIGN KEY)
- `file_name` (usado para verificar duplicados)

### Tabla: `product_details` ⭐ NUEVA

Almacena información detallada de productos asociados a imágenes (dimensiones, peso, precio, SKU).

| Campo | Tipo | Restricciones | Descripción |
|-------|------|---------------|-------------|
| `id` | INTEGER | PRIMARY KEY, AUTO INCREMENT | Identificador único del producto |
| `image_id` | INTEGER | FOREIGN KEY, UNIQUE, NOT NULL | Referencia única a `image.id` (relación 1:1) |
| `alto` | FLOAT | NULLABLE | Altura del producto en centímetros |
| `largo` | FLOAT | NULLABLE | Largo del producto en centímetros |
| `ancho` | FLOAT | NULLABLE | Ancho del producto en centímetros |
| `peso` | FLOAT | NULLABLE | Peso del producto en kilogramos |
| `precio` | FLOAT | NULLABLE | Precio del producto |
| `sku` | VARCHAR(100) | UNIQUE, NULLABLE | Código SKU único del producto (Stock Keeping Unit) |

**Relaciones**:
- One-to-One con `image` (cada producto está asociado a una única imagen)
- Foreign Key: `image_id` → `image.id`
- Cascade: `ondelete="CASCADE"` (si se elimina la imagen, se eliminan sus detalles de producto)
- `unique=True` en `image_id` garantiza la relación 1:1

**Índices**:
- `id` (PRIMARY KEY)
- `image_id` (FOREIGN KEY, UNIQUE)
- `sku` (UNIQUE, para búsquedas rápidas por SKU)

**Nota**: Esta tabla se define en `database_extended.py` y extiende la funcionalidad sin modificar los archivos originales.

## Diagrama de Relaciones

```
┌─────────────────┐
│    category     │
├─────────────────┤
│ id (PK)         │◄────┐
│ name            │     │
│ dir_name        │     │
│ hay_fotos       │     │
│ hay_tamanos     │     │
└─────────────────┘     │
                        │
                        │ (1:N)
                        │
┌─────────────────┐     │
│     image       │     │
├─────────────────┤     │
│ id (PK)         │     │◄────┐
│ ia_name         │     │     │
│ file_name       │     │     │
│ description     │     │     │
│ category_id (FK)├─────┘     │
└─────────────────┘           │
                              │ (1:1)
                              │
┌──────────────────────────┐  │
│    product_details       │  │
├──────────────────────────┤  │
│ id (PK)                  │  │
│ image_id (FK, UNIQUE)    ├──┘
│ alto                     │
│ largo                    │
│ ancho                    │
│ peso                     │
│ precio                   │
│ sku (UNIQUE)             │
└──────────────────────────┘
```

## Operaciones de Base de Datos

### Clase: `Database`

#### Métodos de Categoría

1. **`add_category(name: str, dir_name: str, hay_fotos: bool = True, hay_tamanos: bool = True)`**
   - Crea una nueva categoría
   - Parámetros:
     - `name`: Nombre legible de la categoría
     - `dir_name`: Nombre del directorio
     - `hay_fotos`: Si tiene fotos (default: True)
     - `hay_tamanos`: Si tiene tamaños (default: True)

2. **`get_category(dir_name: str)`**
   - Obtiene una categoría por su `dir_name`
   - Retorna: Objeto `Category` o `None`

3. **`get_all(model: str, type: str = "all")`**
   - Obtiene todas las categorías con filtros opcionales
   - Parámetros:
     - `model`: Debe ser `"Category"`
     - `type`: 
       - `"all"`: Todas las categorías
       - `"pictures"`: Solo categorías con `hay_fotos = True`
       - `"size"`: Solo categorías con `hay_tamanos = True`
   - Retorna: Lista de objetos `Category`

#### Métodos de Imagen

1. **`add_image(ia_name: str, file_name: str, description: str, category_id: int)`**
   - Crea una nueva imagen
   - Parámetros:
     - `ia_name`: Nombre generado por IA
     - `file_name`: Ruta completa del archivo
     - `description`: Descripción del producto (400-800 caracteres)
     - `category_id`: ID de la categoría a la que pertenece

2. **`image_exists(file_name: str)`**
   - Verifica si una imagen ya existe en la base de datos
   - Parámetros:
     - `file_name`: Ruta completa del archivo
   - Retorna: `True` si existe, `False` si no

3. **`count_images(category_id: int)`**
   - Cuenta el número de imágenes en una categoría
   - Parámetros:
     - `category_id`: ID de la categoría
   - Retorna: Número entero

4. **`last_record()`**
   - Obtiene el último registro agregado (útil para debugging)
   - Retorna: Objeto del último registro agregado

### Clase: `DatabaseExtended` ⭐ NUEVA

Clase extendida que proporciona métodos para trabajar con `ProductDetails`. Utiliza la misma conexión de base de datos que `Database`.

**Archivo**: `database_extended.py`

#### Métodos de ProductDetails

1. **`add_product_details(image_id: int, alto: Optional[float] = None, largo: Optional[float] = None, ancho: Optional[float] = None, peso: Optional[float] = None, precio: Optional[float] = None, sku: Optional[str] = None)`**
   - Agrega información de producto para una imagen
   - Parámetros:
     - `image_id`: ID de la imagen a la que se asocia el producto
     - `alto`: Altura en centímetros (opcional)
     - `largo`: Largo en centímetros (opcional)
     - `ancho`: Ancho en centímetros (opcional)
     - `peso`: Peso en kilogramos (opcional)
     - `precio`: Precio del producto (opcional)
     - `sku`: Código SKU único (opcional)
   - Retorna: Objeto `ProductDetails` creado
   - Valida: Que la imagen exista, que no exista ya un producto para esa imagen, y que el SKU sea único

2. **`get_product_details(image_id: int)`**
   - Obtiene los detalles del producto asociados a una imagen
   - Parámetros:
     - `image_id`: ID de la imagen
   - Retorna: Objeto `ProductDetails` o `None` si no existe

3. **`get_product_by_sku(sku: str)`**
   - Obtiene los detalles del producto por su SKU
   - Parámetros:
     - `sku`: Código SKU del producto
   - Retorna: Objeto `ProductDetails` o `None` si no existe

4. **`update_product_details(image_id: int, alto: Optional[float] = None, largo: Optional[float] = None, ancho: Optional[float] = None, peso: Optional[float] = None, precio: Optional[float] = None, sku: Optional[str] = None)`**
   - Actualiza los detalles del producto para una imagen
   - Parámetros: Todos opcionales, solo se actualizan los campos proporcionados
   - Retorna: Objeto `ProductDetails` actualizado o `None` si no existe
   - Valida: Unicidad del SKU si se actualiza

5. **`delete_product_details(image_id: int)`**
   - Elimina los detalles del producto asociados a una imagen
   - Parámetros:
     - `image_id`: ID de la imagen
   - Retorna: `True` si se eliminó, `False` si no existía

6. **`get_all_products_with_details()`**
   - Obtiene todas las imágenes que tienen detalles de producto asociados
   - Retorna: Lista de objetos `ProductDetails` con sus imágenes relacionadas

7. **`get_images_without_product_details()`**
   - Obtiene todas las imágenes que no tienen detalles de producto asociados
   - Retorna: Lista de objetos `Image` sin `ProductDetails`

## Reglas de Negocio

1. **Unicidad de `file_name`**: 
   - Se verifica antes de agregar una imagen para evitar duplicados
   - El campo `file_name` contiene la ruta completa del archivo

2. **Cascada de Eliminación**:
   - Al eliminar una categoría, se eliminan automáticamente todas sus imágenes asociadas

3. **Validación de Categorías**:
   - Las categorías pueden tener solo fotos (`hay_fotos=True`, `hay_tamanos=False`)
   - Las categorías pueden tener solo tamaños (`hay_fotos=False`, `hay_tamanos=True`)
   - Las categorías pueden tener ambos (`hay_fotos=True`, `hay_tamanos=True`)

4. **Descripciones de Imágenes**:
   - Generadas por IA mediante API externa
   - Longitud: entre 400 y 800 caracteres
   - Formato: JSON con `name` y `description`
   - Contenido: Enfoque en formas y características del producto, sin mencionar tamaño

5. **Relación One-to-One Image-ProductDetails**:
   - Cada imagen puede tener como máximo un registro en `product_details`
   - Cada registro en `product_details` está asociado a exactamente una imagen
   - La relación se garantiza mediante `UNIQUE` constraint en `image_id`
   - Al eliminar una imagen, se eliminan automáticamente sus detalles de producto (CASCADE)

6. **Unicidad de SKU**:
   - El campo `sku` debe ser único en toda la tabla si se proporciona
   - Se valida antes de insertar o actualizar
   - Permite valores `NULL` (múltiples productos pueden no tener SKU)

7. **Campos Opcionales en ProductDetails**:
   - Todos los campos de dimensiones y precio son opcionales (NULLABLE)
   - Permite agregar información parcial del producto
   - Se pueden actualizar campos individuales sin afectar los demás

## Ejemplos de Uso

### Crear una categoría
```python
db = Database()
db.add_category(
    name="Figuras Decorativas",
    dir_name="figuras_decorativas",
    hay_fotos=True,
    hay_tamanos=True
)
```

### Obtener categorías con fotos
```python
db = Database()
categorias_con_fotos = db.get_all("Category", type="pictures")
```

### Agregar una imagen
```python
db = Database()
db.add_image(
    ia_name="Ángel Clásico",
    file_name="/ruta/completa/imagen.jpg",
    description="Elegante figura de ángel en yeso...",
    category_id=1
)
```

### Verificar si una imagen existe
```python
db = Database()
if db.image_exists("/ruta/completa/imagen.jpg"):
    print("La imagen ya existe en la base de datos")
```

### Contar imágenes de una categoría
```python
db = Database()
cantidad = db.count_images(category_id=1)
print(f"La categoría tiene {cantidad} imágenes")
```

### Agregar detalles de producto a una imagen ⭐ NUEVO
```python
from database_extended import DatabaseExtended

db_ext = DatabaseExtended()
db_ext.add_product_details(
    image_id=1,
    alto=25.5,
    largo=15.0,
    ancho=10.0,
    peso=0.5,
    precio=29.99,
    sku="YESO-ANG-001"
)
```

### Obtener detalles de producto
```python
from database_extended import DatabaseExtended

db_ext = DatabaseExtended()
producto = db_ext.get_product_details(image_id=1)
if producto:
    print(f"SKU: {producto.sku}, Precio: {producto.precio}")
```

### Buscar producto por SKU
```python
from database_extended import DatabaseExtended

db_ext = DatabaseExtended()
producto = db_ext.get_product_by_sku("YESO-ANG-001")
if producto:
    print(f"Imagen ID: {producto.image_id}")
```

### Actualizar detalles de producto
```python
from database_extended import DatabaseExtended

db_ext = DatabaseExtended()
db_ext.update_product_details(
    image_id=1,
    precio=34.99,  # Solo actualiza el precio
    peso=0.6       # Y el peso
)
```

### Obtener imágenes sin detalles de producto
```python
from database_extended import DatabaseExtended

db_ext = DatabaseExtended()
imagenes_sin_detalles = db_ext.get_images_without_product_details()
print(f"Hay {len(imagenes_sin_detalles)} imágenes sin detalles de producto")
```

## Notas Técnicas

- **SQLAlchemy Echo**: Activado (`echo=True`) para debugging de queries SQL
- **Sesiones**: Se utilizan context managers (`with self.session()`) para manejo automático de transacciones
- **Manejo de Errores**: Todos los métodos tienen try-except con mensajes descriptivos
- **Tipos de Datos**:
  - Strings: VARCHAR con límites (255 para nombres, 1000 para descripciones, 100 para SKU)
  - Booleanos: Tipo `Boolean` de SQLAlchemy
  - Enteros: Auto-incrementales para IDs
  - Floats: Para dimensiones (alto, largo, ancho en cm), peso (en kg) y precio

- **Extensión de Base de Datos**:
  - La nueva tabla `product_details` se define en `database_extended.py`
  - No modifica los archivos originales (`database.py`, `describe.py`)
  - Utiliza la misma base de datos SQLite (`db.sqlite3`)
  - La relación bidireccional con `Image` se maneja desde `database_extended.py`
  - Para usar ambas funcionalidades, importar ambas clases:
    ```python
    from database import Database
    from database_extended import DatabaseExtended
    ```

## Posibles Mejoras Futuras

1. **Índices adicionales**: 
   - Índice en `file_name` para búsquedas más rápidas
   - Índice compuesto en `category_id` y `file_name`

2. **Campos adicionales**:
   - `created_at` y `updated_at` para auditoría
   - `file_size` para almacenar tamaño del archivo
   - `mime_type` para tipo de archivo

3. **Validaciones**:
   - Validar formato de `file_name` (extensión de archivo)
   - Validar longitud de `description` antes de insertar
   - Validar que `category_id` exista antes de insertar imagen

4. **Optimizaciones**:
   - Batch inserts para múltiples imágenes
   - Queries optimizadas con joins explícitos
   - Paginación para `get_all()`

