"""
Ejemplo de uso de la extensión de base de datos (database_extended.py)

Este archivo muestra cómo usar las nuevas funcionalidades para trabajar con
ProductDetails sin modificar los archivos originales del proyecto.
"""

from database import Database
from database_extended import DatabaseExtended


def ejemplo_completo():
    """
    Ejemplo completo de uso de la base de datos extendida.
    """
    # Inicializar ambas clases de base de datos
    db = Database()  # Base de datos original
    db_ext = DatabaseExtended()  # Base de datos extendida
    
    print("=== Ejemplo de uso de DatabaseExtended ===\n")
    
    # 1. Obtener una imagen existente (usando la base de datos original)
    # Asumiendo que ya existe una imagen con id=1
    print("1. Obtener imágenes existentes...")
    categorias = db.get_all("Category", type="pictures")
    if categorias:
        categoria = categorias[0]
        print(f"   Categoría encontrada: {categoria.name} (ID: {categoria.id})")
    
    # 2. Agregar detalles de producto a una imagen
    print("\n2. Agregar detalles de producto...")
    try:
        # Asumiendo que existe una imagen con id=1
        producto = db_ext.add_product_details(
            image_id=1,
            alto=25.5,
            largo=15.0,
            ancho=10.0,
            peso=0.5,
            precio=29.99,
            sku="YESO-ANG-001"
        )
        print(f"   ✓ Producto agregado: SKU={producto.sku}, Precio=${producto.precio}")
    except ValueError as e:
        print(f"   ⚠ Error: {e}")
        print("   (Esto es normal si la imagen no existe o ya tiene detalles)")
    
    # 3. Obtener detalles de producto
    print("\n3. Obtener detalles de producto...")
    producto = db_ext.get_product_details(image_id=1)
    if producto:
        print("   ✓ Producto encontrado:")
        print(f"     - SKU: {producto.sku}")
        print(f"     - Dimensiones: {producto.alto}cm x {producto.largo}cm x {producto.ancho}cm")
        print(f"     - Peso: {producto.peso}kg")
        print(f"     - Precio: ${producto.precio}")
    else:
        print("   ⚠ No se encontraron detalles para esta imagen")
    
    # 4. Buscar producto por SKU
    print("\n4. Buscar producto por SKU...")
    producto = db_ext.get_product_by_sku("YESO-ANG-001")
    if producto:
        print(f"   ✓ Producto encontrado: Image ID={producto.image_id}")
    else:
        print("   ⚠ SKU no encontrado")
    
    # 5. Actualizar detalles de producto
    print("\n5. Actualizar detalles de producto...")
    try:
        producto_actualizado = db_ext.update_product_details(
            image_id=1,
            precio=34.99,  # Solo actualiza el precio
            peso=0.6       # Y el peso
        )
        if producto_actualizado:
            print(f"   ✓ Producto actualizado: Precio=${producto_actualizado.precio}, Peso={producto_actualizado.peso}kg")
        else:
            print("   ⚠ No se encontró el producto para actualizar")
    except ValueError as e:
        print(f"   ⚠ Error: {e}")
    
    # 6. Obtener todas las imágenes con detalles
    print("\n6. Obtener todas las imágenes con detalles...")
    productos = db_ext.get_all_products_with_details()
    print(f"   ✓ Se encontraron {len(productos)} productos con detalles")
    for prod in productos[:3]:  # Mostrar solo los primeros 3
        print(f"     - Image ID: {prod.image_id}, SKU: {prod.sku}")
    
    # 7. Obtener imágenes sin detalles
    print("\n7. Obtener imágenes sin detalles de producto...")
    imagenes_sin_detalles = db_ext.get_images_without_product_details()
    print(f"   ✓ Se encontraron {len(imagenes_sin_detalles)} imágenes sin detalles")
    
    print("\n=== Ejemplo completado ===")


def ejemplo_agregar_producto_completo():
    """
    Ejemplo de cómo agregar un producto completo (imagen + detalles) en un flujo.
    """
    print("\n=== Ejemplo: Agregar producto completo ===\n")
    
    db = Database()
    
    # 1. Primero agregar la imagen (usando la base de datos original)
    print("1. Agregar imagen...")
    try:
        # Obtener una categoría existente
        categorias = db.get_all("Category", type="pictures")
        if not categorias:
            print("   ⚠ No hay categorías disponibles")
            return
        
        
        # Agregar imagen (ejemplo - ajustar según tus necesidades)
        # db.add_image(
        #     ia_name="Ángel Clásico",
        #     file_name="/ruta/imagen.jpg",
        #     description="Elegante figura de ángel...",
        #     category_id=categoria.id
        # )
        # print(f"   ✓ Imagen agregada")
        
        # 2. Obtener el ID de la imagen recién creada
        # (En un caso real, obtendrías el ID del objeto retornado)
        # image_id = imagen.id
        
        # 3. Agregar detalles de producto
        print("2. Agregar detalles de producto...")
        # db_ext.add_product_details(
        #     image_id=image_id,
        #     alto=25.5,
        #     largo=15.0,
        #     ancho=10.0,
        #     peso=0.5,
        #     precio=29.99,
        #     sku="YESO-ANG-001"
        # )
        print("   ✓ Flujo completado (comentado para no modificar datos)")
        
    except Exception as e:
        print(f"   ⚠ Error: {e}")


if __name__ == "__main__":
    # Ejecutar ejemplos
    ejemplo_completo()
    ejemplo_agregar_producto_completo()

