<?php

/**
 * Clase para manejar la conexión y consultas a la base de datos SQLite.
 * Replica la funcionalidad de database.py y database_extended.py
 * 
 * @package Todoyeso\Database
 * @version 1.0
 */
class Database
{
    private ?\PDO $pdo = null;
    private string $dbPath;

    /**
     * Constructor - Inicializa la conexión a la base de datos SQLite
     * 
     * @param string $dbPath Ruta al archivo de base de datos SQLite (por defecto: ./db.sqlite3)
     *                        Puede ser relativa o absoluta. Si es relativa, se resuelve desde el directorio del script.
     * @throws \PDOException Si hay error al conectar
     */
    public function __construct(string $dbPath = './db.sqlite3')
    {
        // Resolver ruta relativa a absoluta si es necesario
        if (!str_starts_with($dbPath, '/')) {
            // Si es relativa, convertir a absoluta basada en el directorio del script
            $resolvedPath = realpath(dirname(__FILE__) . '/../' . $dbPath);
            $this->dbPath = $resolvedPath !== false ? $resolvedPath : $dbPath;
        } else {
            $this->dbPath = $dbPath;
        }
        $this->connect();
    }

    /**
     * Establece la conexión PDO a SQLite
     * 
     * @throws \PDOException Si hay error al conectar
     */
    private function connect(): void
    {
        try {
            $dsn = 'sqlite:' . $this->dbPath;
            $this->pdo = new \PDO($dsn);
            $this->pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
            $this->pdo->setAttribute(\PDO::ATTR_DEFAULT_FETCH_MODE, \PDO::FETCH_ASSOC);
        } catch (\PDOException $e) {
            error_log("Error connecting to database: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene la instancia PDO
     * 
     * @return \PDO
     */
    private function getPdo(): \PDO
    {
        if ($this->pdo === null) {
            $this->connect();
        }
        return $this->pdo;
    }

    // ==================== MÉTODOS DE CATEGORÍA ====================

    /**
     * Agrega una nueva categoría
     * 
     * @param string $name Nombre legible de la categoría
     * @param string $dirName Nombre del directorio asociado
     * @param bool $hayFotos Indica si la categoría tiene fotos (default: true)
     * @param bool $hayTamanos Indica si la categoría tiene tamaños (default: true)
     * @return bool True si se agregó correctamente
     * @throws \PDOException Si hay error al agregar
     */
    public function addCategory(
        string $name,
        string $dirName,
        bool $hayFotos = true,
        bool $hayTamanos = true
    ): bool {
        try {
            $stmt = $this->getPdo()->prepare(
                "INSERT INTO category (name, dir_name, hay_fotos, hay_tamanos) 
                 VALUES (:name, :dir_name, :hay_fotos, :hay_tamanos)"
            );
            
            return $stmt->execute([
                ':name' => $name,
                ':dir_name' => $dirName,
                ':hay_fotos' => $hayFotos ? 1 : 0,
                ':hay_tamanos' => $hayTamanos ? 1 : 0
            ]);
        } catch (\PDOException $e) {
            error_log("Error adding category: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene una categoría por su dir_name
     * 
     * @param string $dirName Nombre del directorio
     * @return array|null Array asociativo con los datos de la categoría o null si no existe
     * @throws \PDOException Si hay error al consultar
     */
    public function getCategory(string $dirName): ?array
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "SELECT * FROM category WHERE dir_name = :dir_name"
            );
            $stmt->execute([':dir_name' => $dirName]);
            $result = $stmt->fetch();
            
            if ($result === false) {
                return null;
            }
            
            // Convertir booleanos
            $result['hay_fotos'] = (bool)$result['hay_fotos'];
            $result['hay_tamanos'] = (bool)$result['hay_tamanos'];
            
            return $result;
        } catch (\PDOException $e) {
            error_log("Error getting category: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene todas las categorías con filtros opcionales
     * 
     * @param string $type Tipo de filtro: "all", "pictures", "size"
     * @return array Lista de categorías
     * @throws \PDOException Si hay error al consultar
     * @throws \InvalidArgumentException Si el tipo es inválido
     */
    public function getAllCategories(string $type = "all"): array
    {
        try {
            $query = "SELECT * FROM category";
            
            if ($type === "pictures") {
                $query .= " WHERE hay_fotos = 1";
            } elseif ($type === "size") {
                $query .= " WHERE hay_tamanos = 1";
            } elseif ($type !== "all") {
                throw new \InvalidArgumentException("Invalid type: {$type}");
            }
            
            $stmt = $this->getPdo()->query($query);
            $results = $stmt->fetchAll();
            
            // Convertir booleanos
            foreach ($results as &$result) {
                $result['hay_fotos'] = (bool)$result['hay_fotos'];
                $result['hay_tamanos'] = (bool)$result['hay_tamanos'];
            }
            
            return $results;
        } catch (\PDOException $e) {
            error_log("Error getting all categories: " . $e->getMessage());
            throw $e;
        }
    }

    // ==================== MÉTODOS DE IMAGEN ====================

    /**
     * Agrega una nueva imagen
     * 
     * @param string $iaName Nombre del producto generado por IA
     * @param string $fileName Ruta completa del archivo de imagen
     * @param string $description Descripción detallada del producto (400-800 caracteres)
     * @param int $categoryId ID de la categoría a la que pertenece
     * @return bool True si se agregó correctamente
     * @throws \PDOException Si hay error al agregar
     */
    public function addImage(
        string $iaName,
        string $fileName,
        string $description,
        int $categoryId
    ): bool {
        try {
            $stmt = $this->getPdo()->prepare(
                "INSERT INTO image (ia_name, file_name, description, category_id) 
                 VALUES (:ia_name, :file_name, :description, :category_id)"
            );
            
            return $stmt->execute([
                ':ia_name' => $iaName,
                ':file_name' => $fileName,
                ':description' => $description,
                ':category_id' => $categoryId
            ]);
        } catch (\PDOException $e) {
            error_log("Error adding image: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Verifica si una imagen ya existe en la base de datos
     * 
     * @param string $fileName Ruta completa del archivo
     * @return bool True si existe, False si no
     * @throws \PDOException Si hay error al consultar
     */
    public function imageExists(string $fileName): bool
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "SELECT COUNT(*) as count FROM image WHERE file_name = :file_name"
            );
            $stmt->execute([':file_name' => $fileName]);
            $result = $stmt->fetch();
            
            return ($result['count'] > 0);
        } catch (\PDOException $e) {
            error_log("Error checking if image exists: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Cuenta el número de imágenes en una categoría
     * 
     * @param int $categoryId ID de la categoría
     * @return int Número de imágenes
     * @throws \PDOException Si hay error al consultar
     */
    public function countImages(int $categoryId): int
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "SELECT COUNT(*) as count FROM image WHERE category_id = :category_id"
            );
            $stmt->execute([':category_id' => $categoryId]);
            $result = $stmt->fetch();
            
            return (int)$result['count'];
        } catch (\PDOException $e) {
            error_log("Error counting images: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene una imagen por su ID
     * 
     * @param int $imageId ID de la imagen
     * @return array|null Array asociativo con los datos de la imagen o null si no existe
     * @throws \PDOException Si hay error al consultar
     */
    public function getImage(int $imageId): ?array
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "SELECT * FROM image WHERE id = :id"
            );
            $stmt->execute([':id' => $imageId]);
            $result = $stmt->fetch();
            
            return $result === false ? null : $result;
        } catch (\PDOException $e) {
            error_log("Error getting image: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene todas las imágenes de una categoría que no tienen detalles de producto
     * 
     * @param int $categoryId ID de la categoría
     * @return array Lista de imágenes sin ProductDetails
     * @throws \PDOException Si hay error al consultar
     */
    public function getImagesWithoutProductDetailsByCategory(int $categoryId): array
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "SELECT i.* 
                 FROM image i 
                 LEFT JOIN product_details pd ON i.id = pd.image_id 
                 WHERE i.category_id = :category_id AND pd.id IS NULL
                 ORDER BY i.id"
            );
            $stmt->execute([':category_id' => $categoryId]);
            return $stmt->fetchAll();
        } catch (\PDOException $e) {
            error_log("Error getting images without product details by category: " . $e->getMessage());
            throw $e;
        }
    }

    // ==================== MÉTODOS DE PRODUCT_DETAILS ====================

    /**
     * Agrega información de producto para una imagen
     * 
     * @param int $imageId ID de la imagen a la que se asocia el producto
     * @param float|null $alto Altura del producto en centímetros
     * @param float|null $largo Largo del producto en centímetros
     * @param float|null $ancho Ancho del producto en centímetros
     * @param float|null $peso Peso del producto en kilogramos
     * @param float|null $precio Precio del producto
     * @param string|null $sku Código SKU único del producto
     * @return bool True si se agregó correctamente
     * @throws \PDOException Si hay error al agregar
     * @throws \InvalidArgumentException Si la imagen no existe o ya tiene detalles
     */
    public function addProductDetails(
        int $imageId,
        ?float $alto = null,
        ?float $largo = null,
        ?float $ancho = null,
        ?float $peso = null,
        ?float $precio = null,
        ?string $sku = null
    ): bool {
        try {
            $pdo = $this->getPdo();
            
            // Verificar que la imagen existe
            $image = $this->getImage($imageId);
            if ($image === null) {
                throw new \InvalidArgumentException("Image with id {$imageId} does not exist");
            }
            
            // Verificar que no existe ya un producto para esta imagen
            $existing = $this->getProductDetails($imageId);
            if ($existing !== null) {
                throw new \InvalidArgumentException("Product details already exist for image_id {$imageId}");
            }
            
            // Verificar unicidad de SKU si se proporciona
            if ($sku !== null && $sku !== '') {
                $existingSku = $this->getProductBySku($sku);
                if ($existingSku !== null) {
                    throw new \InvalidArgumentException("SKU {$sku} already exists");
                }
            }
            
            $stmt = $pdo->prepare(
                "INSERT INTO product_details 
                 (image_id, alto, largo, ancho, peso, precio, sku) 
                 VALUES (:image_id, :alto, :largo, :ancho, :peso, :precio, :sku)"
            );
            
            return $stmt->execute([
                ':image_id' => $imageId,
                ':alto' => $alto,
                ':largo' => $largo,
                ':ancho' => $ancho,
                ':peso' => $peso,
                ':precio' => $precio,
                ':sku' => $sku
            ]);
        } catch (\PDOException $e) {
            error_log("Error adding product details: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene los detalles del producto asociados a una imagen
     * 
     * @param int $imageId ID de la imagen
     * @return array|null Array asociativo con los datos del producto o null si no existe
     * @throws \PDOException Si hay error al consultar
     */
    public function getProductDetails(int $imageId): ?array
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "SELECT * FROM product_details WHERE image_id = :image_id"
            );
            $stmt->execute([':image_id' => $imageId]);
            $result = $stmt->fetch();
            
            return $result === false ? null : $result;
        } catch (\PDOException $e) {
            error_log("Error getting product details: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene los detalles del producto por su SKU
     * 
     * @param string $sku Código SKU del producto
     * @return array|null Array asociativo con los datos del producto o null si no existe
     * @throws \PDOException Si hay error al consultar
     */
    public function getProductBySku(string $sku): ?array
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "SELECT * FROM product_details WHERE sku = :sku"
            );
            $stmt->execute([':sku' => $sku]);
            $result = $stmt->fetch();
            
            return $result === false ? null : $result;
        } catch (\PDOException $e) {
            error_log("Error getting product by SKU: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Actualiza los detalles del producto para una imagen
     * 
     * @param int $imageId ID de la imagen
     * @param float|null $alto Nueva altura (null para no actualizar)
     * @param float|null $largo Nuevo largo (null para no actualizar)
     * @param float|null $ancho Nuevo ancho (null para no actualizar)
     * @param float|null $peso Nuevo peso (null para no actualizar)
     * @param float|null $precio Nuevo precio (null para no actualizar)
     * @param string|null $sku Nuevo SKU (null para no actualizar)
     * @return bool True si se actualizó, False si no existía
     * @throws \PDOException Si hay error al actualizar
     * @throws \InvalidArgumentException Si el SKU ya existe
     */
    public function updateProductDetails(
        int $imageId,
        ?float $alto = null,
        ?float $largo = null,
        ?float $ancho = null,
        ?float $peso = null,
        ?float $precio = null,
        ?string $sku = null
    ): bool {
        try {
            $product = $this->getProductDetails($imageId);
            
            if ($product === null) {
                return false;
            }
            
            // Construir la consulta dinámicamente solo con los campos proporcionados
            $updates = [];
            $params = [':image_id' => $imageId];
            
            if ($alto !== null) {
                $updates[] = "alto = :alto";
                $params[':alto'] = $alto;
            }
            if ($largo !== null) {
                $updates[] = "largo = :largo";
                $params[':largo'] = $largo;
            }
            if ($ancho !== null) {
                $updates[] = "ancho = :ancho";
                $params[':ancho'] = $ancho;
            }
            if ($peso !== null) {
                $updates[] = "peso = :peso";
                $params[':peso'] = $peso;
            }
            if ($precio !== null) {
                $updates[] = "precio = :precio";
                $params[':precio'] = $precio;
            }
            if ($sku !== null) {
                // Verificar unicidad de SKU si se actualiza
                $existingSku = $this->getProductBySku($sku);
                if ($existingSku !== null && $existingSku['id'] != $product['id']) {
                    throw new \InvalidArgumentException("SKU {$sku} already exists");
                }
                $updates[] = "sku = :sku";
                $params[':sku'] = $sku;
            }
            
            if (empty($updates)) {
                return true; // No hay nada que actualizar
            }
            
            $query = "UPDATE product_details SET " . implode(", ", $updates) . 
                     " WHERE image_id = :image_id";
            
            $stmt = $this->getPdo()->prepare($query);
            return $stmt->execute($params);
        } catch (\PDOException $e) {
            error_log("Error updating product details: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Elimina los detalles del producto asociados a una imagen
     * 
     * @param int $imageId ID de la imagen
     * @return bool True si se eliminó, False si no existía
     * @throws \PDOException Si hay error al eliminar
     */
    public function deleteProductDetails(int $imageId): bool
    {
        try {
            $stmt = $this->getPdo()->prepare(
                "DELETE FROM product_details WHERE image_id = :image_id"
            );
            $stmt->execute([':image_id' => $imageId]);
            
            return $stmt->rowCount() > 0;
        } catch (\PDOException $e) {
            error_log("Error deleting product details: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene todas las imágenes que tienen detalles de producto asociados
     * 
     * @return array Lista de productos con sus imágenes relacionadas
     * @throws \PDOException Si hay error al consultar
     */
    public function getAllProductsWithDetails(): array
    {
        try {
            $stmt = $this->getPdo()->query(
                "SELECT pd.*, i.id as image_id_full, i.ia_name, i.file_name, 
                        i.description, i.category_id 
                 FROM product_details pd 
                 INNER JOIN image i ON pd.image_id = i.id"
            );
            
            return $stmt->fetchAll();
        } catch (\PDOException $e) {
            error_log("Error getting all products with details: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Obtiene todas las imágenes que no tienen detalles de producto asociados
     * 
     * @return array Lista de imágenes sin ProductDetails
     * @throws \PDOException Si hay error al consultar
     */
    public function getImagesWithoutProductDetails(): array
    {
        try {
            $stmt = $this->getPdo()->query(
                "SELECT i.* 
                 FROM image i 
                 LEFT JOIN product_details pd ON i.id = pd.image_id 
                 WHERE pd.id IS NULL"
            );
            
            return $stmt->fetchAll();
        } catch (\PDOException $e) {
            error_log("Error getting images without product details: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Cierra la conexión a la base de datos
     */
    public function close(): void
    {
        $this->pdo = null;
    }

    /**
     * Destructor - Cierra la conexión automáticamente
     */
    public function __destruct()
    {
        $this->close();
    }
}

