# Database PHP - Todoyeso

Clase PHP 8 para manejar la conexión y consultas a la base de datos SQLite de Todoyeso.

## Requisitos

- PHP 8.0 o superior
- Extensión PDO con SQLite habilitada

## Instalación

1. Asegúrate de tener PHP 8 instalado:
   ```bash
   php -v
   ```

2. Verifica que la extensión PDO SQLite esté habilitada:
   ```bash
   php -m | grep pdo_sqlite
   ```

3. Copia el archivo `Database.php` a tu proyecto o inclúyelo directamente.

## Uso Básico

```php
<?php
require_once 'Database.php';

// Inicializar la base de datos
$db = new Database('../db.sqlite3');

// Agregar una categoría
$db->addCategory(
    name: "Figuras Decorativas",
    dirName: "figuras_decorativas",
    hayFotos: true,
    hayTamanos: true
);

// Obtener una categoría
$category = $db->getCategory("figuras_decorativas");

// Agregar una imagen
$db->addImage(
    iaName: "Ángel Clásico",
    fileName: "/ruta/completa/imagen.jpg",
    description: "Descripción del producto...",
    categoryId: $category['id']
);

// Agregar detalles de producto
$db->addProductDetails(
    imageId: 1,
    alto: 25.5,
    largo: 15.0,
    ancho: 10.0,
    peso: 0.5,
    precio: 29.99,
    sku: "YESO-ANG-001"
);
```

## Métodos Disponibles

### Categorías

- `addCategory(string $name, string $dirName, bool $hayFotos = true, bool $hayTamanos = true): bool`
- `getCategory(string $dirName): ?array`
- `getAllCategories(string $type = "all"): array`

### Imágenes

- `addImage(string $iaName, string $fileName, string $description, int $categoryId): bool`
- `imageExists(string $fileName): bool`
- `countImages(int $categoryId): int`
- `getImage(int $imageId): ?array`
- `getImagesWithoutProductDetailsByCategory(int $categoryId): array` - Obtiene imágenes de una categoría sin detalles de producto

### Product Details

- `addProductDetails(int $imageId, ?float $alto = null, ?float $largo = null, ?float $ancho = null, ?float $peso = null, ?float $precio = null, ?string $sku = null): bool`
- `getProductDetails(int $imageId): ?array`
- `getProductBySku(string $sku): ?array`
- `updateProductDetails(int $imageId, ?float $alto = null, ...): bool`
- `deleteProductDetails(int $imageId): bool`
- `getAllProductsWithDetails(): array`
- `getImagesWithoutProductDetails(): array`

## Aplicación Web

La carpeta incluye una aplicación web completa para gestionar los detalles de productos:

### Archivos

- **`index.php`**: Página principal que lista todas las categorías
- **`form.php`**: Formulario para agregar detalles de productos a imágenes

### Características

- ✅ Navegación por categorías
- ✅ Muestra solo imágenes sin detalles de producto
- ✅ Formulario con validación (todos los campos requeridos)
- ✅ Interfaz moderna y responsive
- ✅ Indicador de progreso (imágenes pendientes)
- ✅ Redirección automática después de guardar

### Uso

1. Inicia un servidor PHP en la carpeta `php`:
   ```bash
   cd descriptions/php
   php -S localhost:8000
   ```

2. Abre tu navegador en: `http://localhost:8000`

3. Selecciona una categoría y completa el formulario para cada imagen

### Campos del Formulario

Todos los campos son **requeridos**:
- **Alto** (cm): Altura del producto en centímetros
- **Largo** (cm): Largo del producto en centímetros
- **Ancho** (cm): Ancho del producto en centímetros
- **Peso** (kg): Peso del producto en kilogramos
- **Precio**: Precio del producto
- **SKU**: Código único del producto (Stock Keeping Unit)

## Ejemplos

Ver el archivo `ejemplo_uso.php` para ejemplos completos de uso de la clase Database.

## Notas

- La clase utiliza PDO para la conexión a SQLite
- Todos los métodos lanzan excepciones `\PDOException` en caso de error
- Los métodos de validación lanzan `\InvalidArgumentException` cuando los datos son inválidos
- La conexión se cierra automáticamente cuando el objeto se destruye
- Los booleanos se convierten automáticamente entre PHP y SQLite (0/1)

## Compatibilidad

Esta clase es compatible con la estructura de base de datos definida en:
- `database.py`
- `database_extended.py`
- `MODELO_BASE_DATOS.md`

## Diferencias con la versión Python

1. **Tipos de retorno**: Los métodos devuelven arrays asociativos en lugar de objetos
2. **Booleanos**: Se convierten automáticamente entre PHP (true/false) y SQLite (1/0)
3. **Manejo de errores**: Se utilizan excepciones en lugar de try-except con prints
4. **Conexión**: Se utiliza PDO en lugar de SQLAlchemy

