<?php

/**
 * Ejemplo de uso de la clase Database para PHP
 * 
 * Este archivo muestra cómo usar la clase Database para interactuar
 * con la base de datos SQLite de Todoyeso.
 */

require_once __DIR__ . '/Database.php';

// Inicializar la base de datos
// Nota: Ajusta la ruta según la ubicación de tu base de datos
$db = new Database('../db.sqlite3');

try {
    // ==================== EJEMPLOS DE CATEGORÍAS ====================
    
    echo "=== EJEMPLOS DE CATEGORÍAS ===\n\n";
    
    // Agregar una categoría
    echo "1. Agregando categoría...\n";
    $db->addCategory(
        name: "Figuras Decorativas",
        dirName: "figuras_decorativas",
        hayFotos: true,
        hayTamanos: true
    );
    echo "   ✓ Categoría agregada\n\n";
    
    // Obtener una categoría
    echo "2. Obteniendo categoría...\n";
    $category = $db->getCategory("figuras_decorativas");
    if ($category) {
        echo "   ✓ Categoría encontrada: {$category['name']}\n";
        echo "   ID: {$category['id']}\n";
        echo "   Hay fotos: " . ($category['hay_fotos'] ? 'Sí' : 'No') . "\n";
        echo "   Hay tamaños: " . ($category['hay_tamanos'] ? 'Sí' : 'No') . "\n\n";
    }
    
    // Obtener todas las categorías con fotos
    echo "3. Obteniendo categorías con fotos...\n";
    $categoriasConFotos = $db->getAllCategories("pictures");
    echo "   ✓ Encontradas " . count($categoriasConFotos) . " categorías con fotos\n\n";
    
    // ==================== EJEMPLOS DE IMÁGENES ====================
    
    echo "=== EJEMPLOS DE IMÁGENES ===\n\n";
    
    // Verificar si una imagen existe
    echo "4. Verificando si existe una imagen...\n";
    $existe = $db->imageExists("/ruta/completa/imagen.jpg");
    echo "   " . ($existe ? "✓ La imagen existe" : "✗ La imagen no existe") . "\n\n";
    
    // Agregar una imagen (solo si la categoría existe)
    if ($category) {
        echo "5. Agregando imagen...\n";
        $db->addImage(
            iaName: "Ángel Clásico",
            fileName: "/ruta/completa/angel_clasico.jpg",
            description: "Elegante figura de ángel en yeso con detalles artesanales...",
            categoryId: $category['id']
        );
        echo "   ✓ Imagen agregada\n\n";
        
        // Contar imágenes de la categoría
        echo "6. Contando imágenes de la categoría...\n";
        $cantidad = $db->countImages($category['id']);
        echo "   ✓ La categoría tiene {$cantidad} imagen(es)\n\n";
    }
    
    // ==================== EJEMPLOS DE PRODUCT_DETAILS ====================
    
    echo "=== EJEMPLOS DE PRODUCT_DETAILS ===\n\n";
    
    // Obtener una imagen para agregar detalles
    $imagen = $db->getImage(1); // Ajusta el ID según tu base de datos
    
    if ($imagen) {
        echo "7. Agregando detalles de producto...\n";
        try {
            $db->addProductDetails(
                imageId: $imagen['id'],
                alto: 25.5,
                largo: 15.0,
                ancho: 10.0,
                peso: 0.5,
                precio: 29.99,
                sku: "YESO-ANG-001"
            );
            echo "   ✓ Detalles de producto agregados\n\n";
        } catch (Exception $e) {
            echo "   ⚠ " . $e->getMessage() . "\n\n";
        }
        
        // Obtener detalles de producto
        echo "8. Obteniendo detalles de producto...\n";
        $producto = $db->getProductDetails($imagen['id']);
        if ($producto) {
            echo "   ✓ Producto encontrado:\n";
            echo "     SKU: {$producto['sku']}\n";
            echo "     Precio: \${$producto['precio']}\n";
            echo "     Dimensiones: {$producto['alto']} x {$producto['largo']} x {$producto['ancho']} cm\n";
            echo "     Peso: {$producto['peso']} kg\n\n";
        }
        
        // Buscar producto por SKU
        echo "9. Buscando producto por SKU...\n";
        $productoPorSku = $db->getProductBySku("YESO-ANG-001");
        if ($productoPorSku) {
            echo "   ✓ Producto encontrado por SKU:\n";
            echo "     Imagen ID: {$productoPorSku['image_id']}\n";
            echo "     Precio: \${$productoPorSku['precio']}\n\n";
        }
        
        // Actualizar detalles de producto
        echo "10. Actualizando detalles de producto...\n";
        $actualizado = $db->updateProductDetails(
            imageId: $imagen['id'],
            precio: 34.99,  // Solo actualiza el precio
            peso: 0.6       // Y el peso
        );
        if ($actualizado) {
            echo "   ✓ Detalles actualizados\n\n";
        }
    }
    
    // Obtener todos los productos con detalles
    echo "11. Obteniendo todos los productos con detalles...\n";
    $todosLosProductos = $db->getAllProductsWithDetails();
    echo "   ✓ Encontrados " . count($todosLosProductos) . " productos con detalles\n\n";
    
    // Obtener imágenes sin detalles de producto
    echo "12. Obteniendo imágenes sin detalles de producto...\n";
    $imagenesSinDetalles = $db->getImagesWithoutProductDetails();
    echo "   ✓ Encontradas " . count($imagenesSinDetalles) . " imágenes sin detalles\n\n";
    
    echo "=== FIN DE EJEMPLOS ===\n";
    
} catch (Exception $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    echo "Trace: " . $e->getTraceAsString() . "\n";
}

