<?php
/**
 * Formulario para agregar detalles de productos
 * Muestra imágenes de una categoría que no tienen detalles de producto
 */

require_once __DIR__ . '/Database.php';

$db = new Database('../db.sqlite3');

// Obtener parámetros
$categoryId = isset($_GET['category_id']) ? (int)$_GET['category_id'] : 0;
$categoryName = isset($_GET['category_name']) ? htmlspecialchars($_GET['category_name']) : 'Categoría';

// Manejar envío del formulario
$success = false;
$error = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $imageId = (int)$_POST['image_id'];
        $alto = !empty($_POST['alto']) ? (float)$_POST['alto'] : null;
        $largo = !empty($_POST['largo']) ? (float)$_POST['largo'] : null;
        $ancho = !empty($_POST['ancho']) ? (float)$_POST['ancho'] : null;
        $peso = !empty($_POST['peso']) ? (float)$_POST['peso'] : null;
        $precio = !empty($_POST['precio']) ? (float)$_POST['precio'] : null;
        $sku = !empty($_POST['sku']) ? trim($_POST['sku']) : null;

        // Validar que todos los campos estén presentes
        if ($alto === null || $largo === null || $ancho === null || 
            $peso === null || $precio === null || $sku === null || $sku === '') {
            throw new Exception('Todos los campos son requeridos');
        }

        // Validar valores positivos
        if ($alto <= 0 || $largo <= 0 || $ancho <= 0 || $peso <= 0 || $precio <= 0) {
            throw new Exception('Todos los valores numéricos deben ser mayores a cero');
        }

        $db->addProductDetails($imageId, $alto, $largo, $ancho, $peso, $precio, $sku);
        $success = true;
        
        // Redirigir para mostrar la siguiente imagen
        header("Location: form.php?category_id={$categoryId}&category_name=" . urlencode($categoryName) . "&success=1");
        exit;
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Obtener imágenes sin detalles de producto de esta categoría
$images = $db->getImagesWithoutProductDetailsByCategory($categoryId);

// Si hay mensaje de éxito en la URL
$showSuccess = isset($_GET['success']) && $_GET['success'] == '1';

// Obtener la primera imagen si hay alguna
$currentImage = !empty($images) ? $images[0] : null;

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Agregar Detalles - <?php echo $categoryName; ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
        }

        .header {
            background: white;
            padding: 20px 30px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header h1 {
            color: #333;
            font-size: 1.5em;
        }

        .back-link {
            color: #667eea;
            text-decoration: none;
            font-weight: 600;
            padding: 8px 16px;
            border-radius: 5px;
            transition: background 0.3s;
        }

        .back-link:hover {
            background: #f0f0f0;
        }

        .progress {
            background: white;
            padding: 15px 30px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            margin-bottom: 20px;
            text-align: center;
            color: #666;
        }

        .progress strong {
            color: #667eea;
            font-size: 1.2em;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 5px;
            margin-bottom: 20px;
            font-weight: 500;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .form-container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }

        .image-section {
            text-align: center;
            margin-bottom: 30px;
        }

        .image-section img {
            max-width: 100%;
            max-height: 400px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
            margin-bottom: 15px;
        }

        .image-info {
            color: #666;
            margin-top: 10px;
        }

        .image-info h3 {
            color: #333;
            margin-bottom: 5px;
        }

        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }

        .form-group {
            display: flex;
            flex-direction: column;
        }

        .form-group label {
            color: #333;
            font-weight: 600;
            margin-bottom: 8px;
            font-size: 0.9em;
        }

        .form-group input {
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 5px;
            font-size: 1em;
            transition: border-color 0.3s;
        }

        .form-group input:focus {
            outline: none;
            border-color: #667eea;
        }

        .form-group input[type="number"] {
            -moz-appearance: textfield;
        }

        .form-group input[type="number"]::-webkit-outer-spin-button,
        .form-group input[type="number"]::-webkit-inner-spin-button {
            -webkit-appearance: none;
            margin: 0;
        }

        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 25px;
        }

        .btn {
            padding: 12px 30px;
            border: none;
            border-radius: 5px;
            font-size: 1em;
            font-weight: 600;
            cursor: pointer;
            transition: opacity 0.3s, transform 0.2s;
            flex: 1;
        }

        .btn:hover {
            opacity: 0.9;
            transform: translateY(-2px);
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .empty-state {
            background: white;
            padding: 60px 30px;
            border-radius: 10px;
            text-align: center;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }

        .empty-state h2 {
            color: #333;
            margin-bottom: 15px;
        }

        .empty-state p {
            color: #666;
            margin-bottom: 25px;
        }

        .required {
            color: #e74c3c;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><?php echo $categoryName; ?></h1>
            <a href="index.php" class="back-link">← Volver a Categorías</a>
        </div>

        <div class="progress">
            <strong><?php echo count($images); ?></strong> imagen(es) pendiente(s) de detalles
        </div>

        <?php if ($showSuccess): ?>
            <div class="alert alert-success">
                ✓ Detalles de producto guardados correctamente
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-error">
                ✗ Error: <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if (empty($images)): ?>
            <div class="empty-state">
                <h2>🎉 ¡Completado!</h2>
                <p>Todas las imágenes de esta categoría ya tienen detalles de producto.</p>
                <a href="index.php" class="btn btn-primary">Volver a Categorías</a>
            </div>
        <?php elseif ($currentImage): ?>
            <div class="form-container">
                <form method="POST" action="">
                    <input type="hidden" name="image_id" value="<?php echo $currentImage['id']; ?>">
                    
                    <div class="image-section">
                        <?php if (file_exists($currentImage['file_name'])): ?>
                            <img src="<?php echo htmlspecialchars($currentImage['file_name']); ?>" 
                                 alt="<?php echo htmlspecialchars($currentImage['ia_name']); ?>">
                        <?php else: ?>
                            <div style="padding: 100px; background: #f0f0f0; border-radius: 10px; color: #999;">
                                Imagen no encontrada:<br>
                                <small><?php echo htmlspecialchars($currentImage['file_name']); ?></small>
                            </div>
                        <?php endif; ?>
                        <div class="image-info">
                            <h3><?php echo htmlspecialchars($currentImage['ia_name']); ?></h3>
                            <p><?php echo htmlspecialchars(substr($currentImage['description'], 0, 150)) . '...'; ?></p>
                        </div>
                    </div>

                    <div class="form-grid">
                        <div class="form-group">
                            <label for="alto">Alto (cm) <span class="required">*</span></label>
                            <input type="number" 
                                   id="alto" 
                                   name="alto" 
                                   step="0.01" 
                                   min="0.01" 
                                   required 
                                   placeholder="Ej: 25.5">
                        </div>

                        <div class="form-group">
                            <label for="largo">Largo (cm) <span class="required">*</span></label>
                            <input type="number" 
                                   id="largo" 
                                   name="largo" 
                                   step="0.01" 
                                   min="0.01" 
                                   required 
                                   placeholder="Ej: 15.0">
                        </div>

                        <div class="form-group">
                            <label for="ancho">Ancho (cm) <span class="required">*</span></label>
                            <input type="number" 
                                   id="ancho" 
                                   name="ancho" 
                                   step="0.01" 
                                   min="0.01" 
                                   required 
                                   placeholder="Ej: 10.0">
                        </div>

                        <div class="form-group">
                            <label for="peso">Peso (kg) <span class="required">*</span></label>
                            <input type="number" 
                                   id="peso" 
                                   name="peso" 
                                   step="0.01" 
                                   min="0.01" 
                                   required 
                                   placeholder="Ej: 0.5">
                        </div>

                        <div class="form-group">
                            <label for="precio">Precio <span class="required">*</span></label>
                            <input type="number" 
                                   id="precio" 
                                   name="precio" 
                                   step="0.01" 
                                   min="0.01" 
                                   required 
                                   placeholder="Ej: 29.99">
                        </div>

                        <div class="form-group">
                            <label for="sku">SKU <span class="required">*</span></label>
                            <input type="text" 
                                   id="sku" 
                                   name="sku" 
                                   maxlength="100" 
                                   required 
                                   placeholder="Ej: YESO-ANG-001">
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">Guardar Detalles</button>
                        <a href="index.php" class="btn btn-secondary" style="text-decoration: none; display: flex; align-items: center; justify-content: center;">Cancelar</a>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>

